<?php
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;

/**
 * Model Class
 * Handles generic plugin functionality.
 * 
 * @package Easy Digital Downloads - Social Login
 * @since 1.0.0
 */
class EDD_Slg_Model {

	public $foursquare;

	public function __construct() {

		global $edd_slg_social_foursquare;

		$this->foursquare = $edd_slg_social_foursquare;
	}

	/**
	 * Escape Tags & Slashes
	 * Handles escapping the slashes and tags
	 * 
	 * @package  Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_escape_attr( $data ) {
		return esc_attr( stripslashes($data) );
	}

	/**
	 * Strip Slashes From Array
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_escape_slashes_deep( $data = array(), $flag = false ) {
		
		if( $flag != true ) {
			$data = $this->edd_slg_nohtml_kses( $data );
		}

		$data = stripslashes_deep( $data );
		return $data;
	}

	/**
	 * Strip Html Tags
	 * It will sanitize text input (strip html tags, and escape characters)
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_nohtml_kses( $data = array() ) {
		if( is_array($data) ) {
			$data = array_map(array($this, 'edd_slg_nohtml_kses'), $data);
		} elseif( is_string($data) ) {
			$data = wp_filter_nohtml_kses($data);
		}

		return $data;
	}

	/**
	 * Convert Object To Array
	 * Converting Object Type Data To Array Type
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_object_to_array( $result ) {

		$array = array();
		foreach( $result as $key => $value ) {
			if (is_object($value)) {
				$array[$key] = $this->edd_slg_object_to_array( $value );
			} else {
				$array[$key] = $value;
			}
		}
		return $array;
	}

	/**
	 * Create User
	 * 
	 * Function to add connected users to the WordPress users database
	 * and add the role subscriber
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_add_wp_user( $criteria ) {

		global $wp_version, $edd_options;

		$usermeta_prefix = EDD_SLG_USER_META_PREFIX;

		$username = $this->edd_slg_create_username($criteria);

		$name = $criteria['name'];
		$first_name = $criteria['first_name'];
		$last_name = $criteria['last_name'];
		$password = wp_generate_password(12, false);
		$email = $criteria['email'];
		$wp_id = 0;

		// code for register without email
		$criteria['username'] = $username;

		//create the WordPress user
		if( version_compare($wp_version, '3.1', '<') ) {
			require_once( ABSPATH . WPINC . '/registration.php' );
		}

		//check user id is exist or not
		if( $this->edd_slg_check_user_exists($criteria) == false ) {

			if( !empty($email) ) {
				$wp_id = wp_create_user( $username, $password, $email );
			} else { // code for register without email
				$wp_id = wp_create_user( $username, $password );
			}

			//if user is created then update some data
			if( !empty($wp_id) ) {

				$role = get_option( 'default_role' );
				$user = new WP_User( $wp_id );
				$user->set_role( $role );

				//Check if FES plugin is active
				if( class_exists('EDD_Front_End_Submissions') && $role == 'frontend_vendor' ) {

					//Check if FES Registration or Login option enabled
					if( !empty($edd_options['edd_slg_enable_login_options']['edd_slg_fes_register']) || !empty($edd_options['edd_slg_enable_login_options']['edd_slg_fes_login']) ) {

						$db_user = new FES_DB_Vendors();

						$approve_status = EDD_FES()->helper->get_option('fes-auto-approve-vendors');

						$approve_status = (!empty($approve_status) ) ? 'approved' : 'pending';

						if( !$db_user->exists('email', $user->user_email) ) {

							$db_user->add( array(
								'user_id'		=> $user->ID,
								'email'			=> $user->user_email,
								'username'		=> $user->user_login,
								'name'			=> $user->display_name,
								'product_count'	=> 0,
								'status'		=> $approve_status,
							) );
						}
					}
				}

				// call do_action after new user created
				do_action( 'edd_slg_after_create_user', $user->ID, $criteria );

				// check enable email notification from settings
				if( isset($edd_options['edd_slg_enable_notification']) && !empty($edd_options['edd_slg_enable_notification']) ) {
					wp_new_user_notification( $wp_id, null, apply_filters('edd_slg_new_user_notify_to', 'both') );
				}
			}

			//Update unique id to usermeta
			update_user_meta( $wp_id, $usermeta_prefix . 'unique_id', $criteria['id'] );

		} else {

			//get user from email or username
			$userdata = $this->edd_slg_get_user_by( $criteria );

			//check user is exit or not
			if( !empty($userdata) ) {
				$wp_id = isset( $userdata->ID ) ? $userdata->ID : '';
			}
		}
		
		return $wp_id;
	}

	/**
	 * Get Social Connected Users Count
	 * Handles to return connected user counts
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_social_get_users( $args = array() ) {

		$userargs = array();
		$metausr1 = array();

		//check network is set or not
		if( ! empty($args['network']) ) {
			$metausr1['key'] = 'edd_slg_social_user_connect_via';
			$metausr1['value'] = $args['network'];
		}

		//meta query
		if( !empty($metausr1) ) {
			$userargs['meta_query'] = array( $metausr1 );
		}

		//get users data
		$result = new WP_User_Query( $userargs );

		//get count of users
		if( ! empty($args['getcount']) ) {
			$users = $result->total_users;
		} else {
			//retrived data is in object format so assign that data to array for listing
			$users = $this->edd_slg_object_to_array( $users->results );
		}

		return $users;
	}

	/**
	 * Create User Name for VK.com 
	 * Function to check type is vk then create user name based on user id.
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.3.0
	 */
	public function edd_slg_create_username( $criteria ) {

		global $edd_options;

		//Initilize username
		$username = '';

		//Get base of username
		$edd_user_base = isset( $edd_options['edd_slg_base_reg_username'] ) ? $edd_options['edd_slg_base_reg_username'] : '';

		switch( $edd_user_base ) {

			case 'realname' :

				//Get first name
				$first_name = isset( $criteria['first_name'] ) ? strtolower( $criteria['first_name'] ) : '';
				
				//Get last name
				$last_name = isset( $criteria['last_name'] ) ? strtolower( $criteria['last_name'] ) : '';

				//Get username using fname and lname
				$username = $this->edd_slg_username_by_fname_lname( $first_name, $last_name );
				break;

			case 'emailbased' :

				//Get user email
				$user_email = isset( $criteria['email'] ) ? $criteria['email'] : '';

				//Create username using email
				$username = $this->edd_slg_username_by_email( $user_email );
				break;

			default :
				break;
		}

		//If username get empty
		if( empty($username) ) {

			//Get username prefix
			$prefix = EDD_SLG_USER_PREFIX;

			// if service is vk.com then create username with unique id
			if( $criteria['type'] == 'vk' ) {
				$username = $prefix . $criteria['id'];
			} else { // else create create username with random string
				$username = $prefix . wp_rand(100, 9999999);
			}
		}

		//Apply filter to modify username logic
		$username = apply_filters( 'edd_slg_social_username', $username, $criteria );

		//Assign username to temporary variable
		$temp_user_name = $username;

		//Make sure the name is unique: if we've already got a user with this name, append a number to it.
		$counter = 1;

		//If username is exist
		if( username_exists($temp_user_name) ) {
			do {
				$username = $temp_user_name;
				$counter++;
				$username = $username . $counter;
			} while( username_exists($username) );
		} else {
			$username = $temp_user_name;
		}

		return $username;
	}

	/**
	 * Check User Exists
	 * 
	 * Function to check user is exists or not based on either username or email
	 * for VK and 
	 *
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_check_user_exists($criteria) {

		$prefix = EDD_SLG_USER_PREFIX;

		if( ($criteria['type'] == 'vk' && empty($criteria['email'])) ) {
			return $this->edd_slg_user_meta_exists( $criteria['id'] );
		} else {
			if( !empty($criteria['email']) ) {
				return email_exists( $criteria['email'] );
			} else { // code for register without email
				return username_exists( $criteria['username'] );
			}
		}
	}

	/**
	 * User exist from meta
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.5.0
	 */
	public function edd_slg_user_meta_exists( $criteria_id = '', $only_id = true ) {

		//Usermeta prefix
		$user_meta_prefix = EDD_SLG_USER_META_PREFIX;

		$user = array();

		//Get user by meta
		$users = get_users(
			array(
				'meta_key' => $user_meta_prefix . 'unique_id',
				'meta_value' => $criteria_id,
				'number' => 1,
				'count_total' => false
			)
		);

		//If user not empty
		if( !empty($users) ) {
			$user = reset( $users );
		}

		return isset( $user->ID ) ? $user->ID : false;
	}

	/**
	 * User Data By MetaData
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.5.0
	 */
	public function edd_slg_get_user_by_meta( $criteria_id ) {

		//Usermeta prefix
		$user_meta_prefix = EDD_SLG_USER_META_PREFIX;

		$user = array();

		//Get user by meta
		$users = get_users( array(
			'meta_key' => $user_meta_prefix . 'unique_id',
			'meta_value' => $criteria_id,
			'number' => 1,
			'count_total' => false
		) );

		//If user not empty
		if( !empty($users) ) {
			$user = reset( $users );
		}

		return $user;
	}

	/**
	 * Get User by email or username
	 * 
	 * Function to get user by email or username
	 * for VK 
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_get_user_by( $criteria ) {

		$prefix = EDD_SLG_USER_PREFIX;

		if( $criteria['type'] == 'vk' && empty($criteria['email']) ) {
			return $this->edd_slg_get_user_by_meta( $criteria['id'] );
		} else {
			return get_user_by( 'email', $criteria['email'] );
		}
	}

	/**
	 * Add plugin section in extension settings
	 *
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.7.2
	 */
	public function edd_slg_settings_section($sections) {
		$sections['eddslg'] = esc_html__( 'Social Login', 'eddslg' );
		return $sections;
	}

	/**
	 * Register Settings
	 * 
	 * Handels to add settings in settings page
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.0.0
	 */
	public function edd_slg_settings( $settings ) {

		$success_message = '';
		// Display success message when click reset social setting
		if( isset($_GET['settings-updated']) && $_GET['settings-updated'] == 'edd_slg_reset' ) {
			$success_message = '<div class="updated" id="message"><p><strong>' . esc_html__( 'Social login settings reset successfully.', 'eddslg' ) . '</strong></p></div>';
		}

		$select_fblanguage = array(
			'en_US' => esc_html__( 'English', 'eddslg' ),
			'af_ZA' => esc_html__( 'Afrikaans', 'eddslg' ),
			'sq_AL' => esc_html__( 'Albanian', 'eddslg' ),
			'ar_AR' => esc_html__( 'Arabic', 'eddslg' ),
			'hy_AM' => esc_html__( 'Armenian', 'eddslg' ),
			'eu_ES' => esc_html__( 'Basque', 'eddslg' ),
			'be_BY' => esc_html__( 'Belarusian', 'eddslg' ),
			'bn_IN' => esc_html__( 'Bengali', 'eddslg' ),
			'bs_BA' => esc_html__( 'Bosanski', 'eddslg' ),
			'bg_BG' => esc_html__( 'Bulgarian', 'eddslg' ),
			'ca_ES' => esc_html__( 'Catalan', 'eddslg' ),
			'zh_CN' => esc_html__( 'Chinese', 'eddslg' ),
			'cs_CZ' => esc_html__( 'Czech', 'eddslg' ),
			'da_DK' => esc_html__( 'Danish', 'eddslg' ),
			'fy_NL' => esc_html__( 'Dutch', 'eddslg' ),
			'eo_EO' => esc_html__( 'Esperanto', 'eddslg' ),
			'et_EE' => esc_html__( 'Estonian', 'eddslg' ),
			'et_EE' => esc_html__( 'Estonian', 'eddslg' ),
			'fi_FI' => esc_html__( 'Finnish', 'eddslg' ),
			'fo_FO' => esc_html__( 'Faroese', 'eddslg' ),
			'tl_PH' => esc_html__( 'Filipino', 'eddslg' ),
			'fr_FR' => esc_html__( 'French', 'eddslg' ),
			'gl_ES' => esc_html__( 'Galician', 'eddslg' ),
			'ka_GE' => esc_html__( 'Georgian', 'eddslg' ),
			'de_DE' => esc_html__( 'German', 'eddslg' ),
			'zh_CN' => esc_html__( 'Greek', 'eddslg' ),
			'he_IL' => esc_html__( 'Hebrew', 'eddslg' ),
			'hi_IN' => esc_html__( 'Hindi', 'eddslg' ),
			'hr_HR' => esc_html__( 'Hrvatski', 'eddslg' ),
			'hu_HU' => esc_html__( 'Hungarian', 'eddslg' ),
			'is_IS' => esc_html__( 'Icelandic', 'eddslg' ),
			'id_ID' => esc_html__( 'Indonesian', 'eddslg' ),
			'ga_IE' => esc_html__( 'Irish', 'eddslg' ),
			'it_IT' => esc_html__( 'Italian', 'eddslg' ),
			'ja_JP' => esc_html__( 'Japanese', 'eddslg' ),
			'ko_KR' => esc_html__( 'Korean', 'eddslg' ),
			'ku_TR' => esc_html__( 'Kurdish', 'eddslg' ),
			'la_VA' => esc_html__( 'Latin', 'eddslg' ),
			'lv_LV' => esc_html__( 'Latvian', 'eddslg' ),
			'fb_LT' => esc_html__('Leet  Speak', 'eddslg' ),
			'lt_LT' => esc_html__( 'Lithuanian', 'eddslg' ),
			'mk_MK' => esc_html__( 'Macedonian', 'eddslg' ),
			'ms_MY' => esc_html__( 'Malay', 'eddslg' ),
			'ml_IN' => esc_html__( 'Malayalam', 'eddslg' ),
			'nl_NL' => esc_html__( 'Nederlands', 'eddslg' ),
			'ne_NP' => esc_html__( 'Nepali', 'eddslg' ),
			'nb_NO' => esc_html__( 'Norwegian', 'eddslg' ),
			'ps_AF' => esc_html__( 'Pashto', 'eddslg' ),
			'fa_IR' => esc_html__( 'Persian', 'eddslg' ),
			'pl_PL' => esc_html__( 'Polish', 'eddslg' ),
			'pt_PT' => esc_html__( 'Portugese', 'eddslg' ),
			'pa_IN' => esc_html__( 'Punjabi', 'eddslg' ),
			'ro_RO' => esc_html__( 'Romanian', 'eddslg' ),
			'ru_RU' => esc_html__( 'Russian', 'eddslg' ),
			'sk_SK' => esc_html__( 'Slovak', 'eddslg' ),
			'sl_SI' => esc_html__( 'Slovenian', 'eddslg' ),
			'es_LA' => esc_html__( 'Spanish', 'eddslg' ),
			'sr_RS' => esc_html__( 'Srpski', 'eddslg' ),
			'sw_KE' => esc_html__( 'Swahili', 'eddslg' ),
			'sv_SE' => esc_html__( 'Swedish', 'eddslg' ),
			'ta_IN' => esc_html__( 'Tamil', 'eddslg' ),
			'te_IN' => esc_html__( 'Telugu', 'eddslg' ),
			'th_TH' => esc_html__( 'Thai', 'eddslg' ),
			'tr_TR' => esc_html__( 'Turkish', 'eddslg' ),
			'uk_UA' => esc_html__( 'Ukrainian', 'eddslg' ),
			'vi_VN' => esc_html__( 'Vietnamese', 'eddslg' ),
			'cy_GB' => esc_html__( 'Welsh', 'eddslg' ),
			'zh_TW' => esc_html__( 'Traditional Chinese Language', 'eddslg' )
		);

		$select_scope = array(
			'signin' => esc_html__( 'Sign In', 'eddslg' ),
			'openid' => esc_html__( 'Open ID', 'eddslg' )
		);

		$positions = array(
			'top'		=> esc_html__('Above Login/Register form', 'eddslg'),
			'bottom'	=> esc_html__('Below Login/Register form', 'eddslg'),
			'hook'		=> esc_html__('Custom Hook', 'eddslg')
		);

		$edd_slg_settings = array(
			array(
				'id' => 'edd_slg_settings',
				'name' => $success_message . '<strong>' . esc_html__('Social Login Options', 'eddslg') . '</strong>',
				'desc' => '<input type="button" value="Reset Setting" />' . esc_html__('Configure Social Login Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_social_reset',
				'name' => '<strong>' . esc_html__('Reset Settings', 'eddslg') . '</strong>',
				'desc' => '<p class="description">' . esc_html__('This will reset all the setings of social login.', 'eddslg') . '</p>',
				'type' => 'social_reset',
				'size' => 'button',
			),
			//General Settings
			array(
				'id' => 'edd_slg_general_settings',
				'name' => '<strong>' . esc_html__('General Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login General Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_login_heading',
				'name' => esc_html__('Social Login Title:', 'eddslg'),
				'desc' => esc_html__('Enter Social Login Title.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_login_options',
				'name' => esc_html__('Display Social Login buttons on:', 'eddslg'),
				'type' => 'multicheck',
				'options' => apply_filters('edd_slg_enable_login_options', array(
					'edd_login' => esc_html__('Check this box to add social login buttons on Easy Digital Downloads login page.', 'eddslg'),
					'edd_register' => esc_html__('Check this box to add social login buttons on Easy Digital Downloads register page.', 'eddslg'),
					'edd_checkout' => esc_html__('Check this box to add social login buttons on Easy Digital Downloads checkout page.', 'eddslg'),
					'wp_login' => esc_html__('Check this box to add social login buttons on default WordPress login page.', 'eddslg'),
					'wp_register' => esc_html__('Check this box to add social login buttons on default WordPress register page.', 'eddslg'),
					)
				)
			),
			array(
				'id' => 'edd_slg_display_link_thank_you',
				'name' => esc_html__('Display "Link Your Account" button on Thank You page:', 'eddslg'),
				'type' => 'checkbox',
				'desc' => '<p class="description">' . esc_html__(' Check this box to allow customers to link their social account on the Thank You page for faster login & checkout next time they purchase.', 'eddslg') . '</p>'
			),
			array(
				'id' => 'edd_slg_social_btn_position',
				'name' => esc_html__('Social Buttons Position:', 'eddslg'),
				'type' => 'checkbox',
				'desc' => '<p class="description">' . esc_html__(' Select the postion where you want to display the social login buttons. Select Custom Hook if you want to display on custom form.', 'eddslg') . '</p>',
				'type' => 'select',
				'options' => $positions,
			),
			array(
				'id' => 'edd_slg_social_btn_hooks',
				'name' => esc_html__('Custom Hooks:', 'eddslg'),
				'desc' => '<p class="description">' . sprintf(esc_html__('If your theme/plugin does provide custom hooks then you can enter the name of the hook here. You can find more information about using hook in the %s manual %s.', 'eddslg'), '<a href="https://docs.wpwebelite.com/edd-social-login/social-login-setup-docs/#using-custom-hooks" target="_blank">', '</a>') . '</p>',
				'type' => 'addmore',
				'size' => 'regular',
			),
			array(
				'id' => 'edd_slg_enable_notification',
				'name' => esc_html__('Enable Email Notification:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to notify admin and user when user is registered by social media.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_redirect_url',
				'name' => esc_html__('Redirect URL:', 'eddslg'),
				'desc' => esc_html__('Enter a redirect URL for users after they login with social media. The URL must start with', 'eddslg') . ' http://',
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_base_reg_username',
				'name' => esc_html__('Autoregistered Usernames:', 'eddslg'),
				'desc' => '',
				'type' => 'radio',
				'size' => 'regular',
				'std' => '',
				'options' => array(
					'' => esc_html__('Based on unique ID & random number (i.e. edd_slg_123456)', 'eddslg'),
					'realname' => esc_html__('Based on real name (i.e. john_smith)', 'eddslg'),
					'emailbased' => esc_html__('Based on email ID (i.e. john.smith@example.com to john_smith_example_com )', 'eddslg')
				)
			),
			array(
				'id' => 'edd_slg_enable_caching_resolution',
				'name' => esc_html__('Caching:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you use a caching plugin.', 'eddslg'),
				'type' => 'checkbox'
			),
			//Facebbok Settings
			array(
				'id' => 'edd_slg_facebook_settings',
				'name' => '<strong>' . esc_html__('Facebook Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Facebook Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_facebook_desc',
				'name' => esc_html__('Facebook Application:', 'eddslg'),
				'desc' => '',
				'type' => 'facebook_desc'
			),
			array(
				'id' => 'edd_slg_enable_facebook',
				'name' => esc_html__('Enable Facebook:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable facebook social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_fb_app_id',
				'name' => esc_html__('Facebook App ID/API Key:', 'eddslg'),
				'desc' => esc_html__('Enter Facebook API Key.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_fb_app_secret',
				'name' => esc_html__('Facebook App Secret:', 'eddslg'),
				'desc' => esc_html__('Enter Facebook App Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_fb_redirect_url',
				'name' => esc_html__('Facebook Valid OAuth Redirect URL:', 'eddslg'),
				'desc' => '',
				'type' => 'fb_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_fb_language',
				'name' => esc_html__('Facebook API Locale:', 'eddslg'),
				'desc' => esc_html__('Select the language for Facebook. With this option, you can explicitly tell which language you want to use for communicating with Facebook.', 'eddslg'),
				'type' => 'select',
				'options' => $select_fblanguage
			),
			array(
				'id' => 'edd_slg_fb_icon_url',
				'name' => esc_html__('Custom Facebook Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Facebook Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_fb_link_icon_url',
				'name' => esc_html__('Custom Facebook Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Facebook Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_fb_avatar',
				'name' => esc_html__('Enable Facebook Avatar:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to use Facebook profile pictures as avatars.', 'eddslg'),
				'type' => 'checkbox'
			),
			//Google+ Settings
			array(
				'id' => 'edd_slg_googleplus_settings',
				'name' => '<strong>' . esc_html__('Google Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Google Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_googleplus_desc',
				'name' => esc_html__('Google Application:', 'eddslg'),
				'desc' => '',
				'type' => 'googleplus_desc'
			),
			array(
				'id' => 'edd_slg_enable_googleplus',
				'name' => esc_html__('Enable Google:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable google social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_gp_client_id',
				'name' => esc_html__('Google Client ID:', 'eddslg'),
				'desc' => esc_html__('Enter Google Client ID.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_gp_redirect_url',
				'name' => esc_html__('Google JavaScript origins URL:', 'eddslg'),
				'desc' => '',
				'type' => 'gp_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_gp_redirect_url_notice',
				'name' => '',
				'desc' => '',
				'type' => 'gp_redirect_url_notice',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_gp_icon_url',
				'name' => esc_html__('Custom Google Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Google Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_gp_link_icon_url',
				'name' => esc_html__('Custom Google Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Google Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_gp_avatar',
				'name' => esc_html__('Enable Google Plus Avatar:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to use Google Plus profile pictures as avatars.', 'eddslg'),
				'type' => 'checkbox'
			),
			//LinkedIn Settings
			array(
				'id' => 'edd_slg_linkedin_settings',
				'name' => '<strong>' . esc_html__('LinkedIn Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login LinkedIn Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_linkedin_desc',
				'name' => esc_html__('LinkedIn Application:', 'eddslg'),
				'desc' => '',
				'type' => 'linkedin_desc'
			),
			array(
				'id' => 'edd_slg_enable_linkedin',
				'name' => esc_html__('Enable LinkedIn:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable LinkedIn social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_li_scope_type',
				'name' => esc_html__('Additional Product Access Option :', 'eddslg'),
				'desc' => esc_html__('Select OpenID Connect option for newly created APP. SignIn option is for backward compatibility for old APP users only. ', 'eddslg'),
				'type' => 'select',
				'options' => $select_scope
			),
			array(
				'id' => 'edd_slg_li_app_id',
				'name' => esc_html__('LinkedIn App ID/API Key:', 'eddslg'),
				'desc' => esc_html__('Enter LinkedIn App ID/API Key.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_li_app_secret',
				'name' => esc_html__('LinkedIn App Secret:', 'eddslg'),
				'desc' => esc_html__('Enter LinkedIn App Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_li_redirect_url',
				'name' => esc_html__('Linkedin Redirect URI:', 'eddslg'),
				'desc' => '',
				'type' => 'li_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_li_icon_url',
				'name' => esc_html__('Custom LinkedIn Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own LinkedIn Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_li_link_icon_url',
				'name' => esc_html__('Custom LinkedIn Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own LinkedIn Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_li_avatar',
				'name' => esc_html__('Enable LinkedIn Avatar:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to use LinkedIn profile pictures as avatars.', 'eddslg'),
				'type' => 'checkbox'
			),
			//twitter Settings
			array(
				'id' => 'edd_slg_twitter_settings',
				'name' => '<strong>' . esc_html__('Twitter Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Twitter Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_twitter_desc',
				'name' => esc_html__('Twitter Application:', 'eddslg'),
				'desc' => '',
				'type' => 'twitter_desc'
			),
			array(
				'id' => 'edd_slg_enable_twitter',
				'name' => esc_html__('Enable Twitter:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable Twitter social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_tw_consumer_key',
				'name' => esc_html__('Twitter API Key:', 'eddslg'),
				'desc' => esc_html__('Enter Twitter API Key.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_tw_consumer_secret',
				'name' => esc_html__('Twitter API Secret:', 'eddslg'),
				'desc' => esc_html__('Enter Twitter API Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_tw_icon_url',
				'name' => esc_html__('Custom Twitter Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Twitter Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_tw_link_icon_url',
				'name' => esc_html__('Custom Twitter Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Twitter Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_tw_avatar',
				'name' => esc_html__('Enable Twitter Avatar:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to use Twitter profile pictures as avatars.', 'eddslg'),
				'type' => 'checkbox'
			),
			//yahoo Settings
			array(
				'id' => 'edd_slg_yahoo_settings',
				'name' => '<strong>' . esc_html__('Yahoo Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Yahoo Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_yahoo_desc',
				'name' => esc_html__('Yahoo Application:', 'eddslg'),
				'desc' => '',
				'type' => 'yahoo_desc'
			),
			array(
				'id' => 'edd_slg_enable_yahoo',
				'name' => esc_html__('Enable Yahoo:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable Yahoo social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_yh_consumer_key',
				'name' => esc_html__('Yahoo Consumer Key:', 'eddslg'),
				'desc' => esc_html__('Enter Yahoo Consumer Key.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_yh_consumer_secret',
				'name' => esc_html__('Yahoo Consumer Secret:', 'eddslg'),
				'desc' => esc_html__('Enter Yahoo Consumer Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_yh_redirect_url',
				'name' => esc_html__('Yahoo Callback URL:', 'eddslg'),
				'desc' => '',
				'type' => 'yh_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_yh_icon_url',
				'name' => esc_html__('Custom Yahoo Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Yahoo Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_yh_link_icon_url',
				'name' => esc_html__('Custom Yahoo Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Yahoo Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_yh_avatar',
				'name' => esc_html__('Enable Yahoo Avatar:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to use Yahoo profile pictures as avatars.', 'eddslg'),
				'type' => 'checkbox'
			),
			//Foursquare Settings
			array(
				'id' => 'edd_slg_foursquare_settings',
				'name' => '<strong>' . esc_html__('Foursquare Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Foursquare Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_foursquare_desc',
				'name' => esc_html__('Foursquare Application:', 'eddslg'),
				'desc' => '',
				'type' => 'foursquare_desc'
			),
			array(
				'id' => 'edd_slg_enable_foursquare',
				'name' => esc_html__('Enable Foursquare:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable Foursquare social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_fs_client_id',
				'name' => esc_html__('Foursquare Client ID:', 'eddslg'),
				'desc' => esc_html__('Enter Foursquare Client ID.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_fs_client_secret',
				'name' => esc_html__('Foursquare Client Secret:', 'eddslg'),
				'desc' => esc_html__('Enter Foursquare Client Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_fs_icon_url',
				'name' => esc_html__('Custom Foursquare Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Foursquare Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_fs_link_icon_url',
				'name' => esc_html__('Custom Foursquare Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Foursquare Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_fs_avatar',
				'name' => esc_html__('Enable Foursquare Avatar:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to use Foursquare profile pictures as avatars.', 'eddslg'),
				'type' => 'checkbox'
			),
			//Windows Live Settings
			array(
				'id' => 'edd_slg_windowslive_settings',
				'name' => '<strong>' . esc_html__('Windows Live Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Windows Live Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_windowslive_desc',
				'name' => esc_html__('Windows Live Application:', 'eddslg'),
				'desc' => '',
				'type' => 'windowslive_desc'
			),
			array(
				'id' => 'edd_slg_enable_windowslive',
				'name' => esc_html__('Enable Windows Live:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable Windows Live social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_wl_client_id',
				'name' => esc_html__('Windows Live Client ID:', 'eddslg'),
				'desc' => esc_html__('Enter Windows Live Client ID.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_wl_client_secret',
				'name' => esc_html__('Windows Live Client Secret:', 'eddslg'),
				'desc' => esc_html__('Enter Windows Live Client Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_wl_redirect_url',
				'name' => esc_html__('Windows Live Callback URL:', 'eddslg'),
				'desc' => '',
				'type' => 'wl_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_wl_icon_url',
				'name' => esc_html__('Custom Windows Live Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Windows Live Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_wl_link_icon_url',
				'name' => esc_html__('Custom Windows Live Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Windows Live Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			//VK Settings
			array(
				'id' => 'edd_slg_vk_settings',
				'name' => '<strong>' . esc_html__('VK Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login VK Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_vk_desc',
				'name' => esc_html__('VK Application:', 'eddslg'),
				'desc' => '',
				'type' => 'vk_desc'
			),
			array(
				'id' => 'edd_slg_enable_vk',
				'name' => esc_html__('Enable VK:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable vk social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_vk_app_id',
				'name' => esc_html__('VK Application ID:', 'eddslg'),
				'desc' => esc_html__('Enter VK Application ID.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_vk_app_secret',
				'name' => esc_html__('VK Secret Key:', 'eddslg'),
				'desc' => esc_html__('Enter VK Secret Key.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_vk_icon_url',
				'name' => esc_html__('Custom VK Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own VK Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_vk_link_icon_url',
				'name' => esc_html__('Custom VK Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own VK Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_enable_vk_avatar',
				'name' => esc_html__('Enable VK Avatar:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to use VK profile pictures as avatars.', 'eddslg'),
				'type' => 'checkbox'
			),
			//amazon Settings
			array(
				'id' => 'edd_slg_amazon_settings',
				'name' => '<strong>' . esc_html__('Amazon Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Amazon Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_amazon_desc',
				'name' => esc_html__('Amazon Application:', 'eddslg'),
				'desc' => '',
				'type' => 'amazon_desc'
			),
			array(
				'id' => 'edd_slg_enable_amazon',
				'name' => esc_html__('Enable Amazon:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable amazon social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_amazon_app_id',
				'name' => esc_html__('Amazon Client ID:', 'eddslg'),
				'desc' => esc_html__('Enter Amazon Client ID.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_amazon_app_secret',
				'name' => esc_html__('Amazon Client Secret:', 'eddslg'),
				'desc' => esc_html__('Enter Amazon Client Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_amazon_redirect_url',
				'name' => esc_html__('Amazon Callback URL:', 'eddslg'),
				'desc' => '',
				'type' => 'amazon_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_amazon_icon_url',
				'name' => esc_html__('Custom Amazon Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Amazon Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_amazon_link_icon_url',
				'name' => esc_html__('Custom Amazon Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Amazon Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			//paypal Settings
			array(
				'id' => 'edd_slg_paypal_settings',
				'name' => '<strong>' . esc_html__('Paypal Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Paypal Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_paypal_desc',
				'name' => esc_html__('Paypal Application:', 'eddslg'),
				'desc' => '',
				'type' => 'paypal_desc'
			),
			array(
				'id' => 'edd_slg_enable_paypal',
				'name' => esc_html__('Enable Paypal:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable paypal social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_paypal_app_id',
				'name' => esc_html__('Paypal Client ID:', 'eddslg'),
				'desc' => esc_html__('Enter Paypal Client ID.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_paypal_app_secret',
				'name' => esc_html__('Paypal Client Secret:', 'eddslg'),
				'desc' => esc_html__('Enter Paypal Client Secret.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_paypal_redirect_url',
				'name' => esc_html__('Paypal Callback URL:', 'eddslg'),
				'desc' => '',
				'type' => 'paypal_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_paypal_icon_url',
				'name' => esc_html__('Custom Paypal Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Paypal Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_paypal_link_icon_url',
				'name' => esc_html__('Custom Paypal Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Paypal Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_paypal_environment',
				'name' => esc_html__('Environment:', 'eddslg'),
				'desc' => esc_html__('Select which environment to process logins under.', 'eddslg'),
				'type' => 'select',
				'options' => array('live' => esc_html__('Live', 'eddslg'), 'sandbox' => esc_html__('Sandbox', 'eddslg'))
			),
			//Apple Settings
			array(
				'id' => 'edd_slg_apple_settings',
				'name' => '<strong>' . esc_html__('Apple Settings', 'eddslg') . '</strong>',
				'desc' => esc_html__('Configure Social Login Apple Settings', 'eddslg'),
				'type' => 'header'
			),
			array(
				'id' => 'edd_slg_apple_desc',
				'name' => esc_html__('Apple Application:', 'eddslg'),
				'desc' => '',
				'type' => 'apple_desc'
			),
			array(
				'id' => 'edd_slg_enable_apple',
				'name' => esc_html__('Enable Apple:', 'eddslg'),
				'desc' => esc_html__('Check this box, if you want to enable apple social login registration.', 'eddslg'),
				'type' => 'checkbox'
			),
			array(
				'id' => 'edd_slg_apple_client_id',
				'name' => esc_html__('Apple Client ID:', 'eddslg'),
				'desc' => esc_html__('Enter Apple Client ID.', 'eddslg'),
				'type' => 'text',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_apple_redirect_url',
				'name' => esc_html__('Apple Valid Redirect URL:', 'eddslg'),
				'desc' => '',
				'type' => 'apple_redirect_url',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_apple_icon_url',
				'name' => esc_html__('Custom Apple Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Apple Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			),
			array(
				'id' => 'edd_slg_apple_link_icon_url',
				'name' => esc_html__('Custom Apple Link Icon:', 'eddslg'),
				'desc' => esc_html__('If you want to use your own Apple Link Icon, upload one here.', 'eddslg'),
				'type' => 'upload',
				'size' => 'regular'
			)
		);

		// If EDD is at version 2.5 or later
		if( version_compare(EDD_VERSION, 2.5, '>=') ) {
			$edd_slg_settings = array( 'eddslg' => $edd_slg_settings );
		}

		return array_merge($settings, $edd_slg_settings);
	}

	/**
	 * Get User profile pic
	 *
	 * Function to get user profile pic from user meta type its social type 
	 *
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.4
	 */
	public function edd_slg_get_user_profile_pic( $user_id = false ) {

		global $edd_options;

		// Taking some defaults
		$profile_pic_url = '';

		// If user id is passed then take otherwise take current user
		$user_id = !empty($user_id) ? $user_id : '';

		if( $user_id ) {

			// Getting some user details
			$edd_slg_social_type = get_user_meta( $user_id, 'edd_slg_social_user_connect_via', true );
			$edd_slg_data = get_user_meta( $user_id, 'edd_slg_social_data', true );

			if( !empty($edd_slg_social_type) && !empty($edd_slg_data) ) {

				// If facebook avatar is enable
				if( !empty($edd_options['edd_slg_enable_fb_avatar']) && 
					$edd_options['edd_slg_enable_fb_avatar'] == "1" ) {

					// If user is from facebook
					if( $edd_slg_social_type == 'facebook' ) {
						$profile_pic_url = !empty( $edd_slg_data['picture'] ) ? $edd_slg_data['picture'] : '';
					}
				}

				// If twitter avatar is enable
				if( !empty($edd_options['edd_slg_enable_tw_avatar']) && 
					$edd_options['edd_slg_enable_tw_avatar'] == "1" ) {

					// If user is from twitter
					if( $edd_slg_social_type == 'twitter' ) {
						$profile_pic_url = !empty($edd_slg_data->profile_image_url_https) ? $edd_slg_data->profile_image_url_https : '';
					}
				}

				// If google plus avatar is enable
				if( !empty($edd_options['edd_slg_enable_gp_avatar']) && 
					$edd_options['edd_slg_enable_gp_avatar'] == "1" ) {

					// If user is from googleplus
					if( $edd_slg_social_type == 'googleplus' ) {
						if( isset($edd_slg_data->picture) && !empty($edd_slg_data->picture) ) {
							$profile_pic_url = $edd_slg_data->picture;

						// Added for backward compitibility
						} elseif( $edd_slg_data['image']['url'] && !empty($edd_slg_data['image']['url']) ) {
							$profile_pic_url = $edd_slg_data['image']['url'];
						}
					}
				}

				// If linked in avatar is enable
				if( !empty($edd_options['edd_slg_enable_li_avatar']) && 
					$edd_options['edd_slg_enable_li_avatar'] == "1" ) {

					// If user is from linkedin
					if( $edd_slg_social_type == 'linkedin' ) {

						$profile_pic_url = '';

						// Added for backward compitibility
						if( !empty($edd_slg_data['picture-url']) ) {
							$profile_pic_url = $edd_slg_data['picture-url'];
						} elseif( !empty($edd_slg_data['pictureUrl']) ) {
							$profile_pic_url = $edd_slg_data['pictureUrl'];
						}
					}
				}

				// If yahoo avatar is enable
				if( !empty($edd_options['edd_slg_enable_yh_avatar']) && 
					$edd_options['edd_slg_enable_yh_avatar'] == "1" ) {

					// If user is from yahoo
					if( $edd_slg_social_type == 'yahoo' ) {
						$profile_pic_url = !empty( $edd_slg_data['image']['imageUrl'] ) ? $edd_slg_data['image']['imageUrl'] : '';
					}
				}

				// If foursquer avatar is enable
				if( !empty($edd_options['edd_slg_enable_fs_avatar']) && 
					$edd_options['edd_slg_enable_fs_avatar'] == "1" ) {

					// If user is from foursquare
					if( $edd_slg_social_type == 'foursquare' ) {
						$profile_pic_url = $this->foursquare->edd_slg_get_foursquare_profile_picture( array('size' => '64'), $edd_slg_data );
					}
				}

				// If vk avatar is enable
				if( !empty($edd_options['edd_slg_enable_vk_avatar']) && 
					$edd_options['edd_slg_enable_vk_avatar'] == "1" ) {

					// If user is from vk
					if( $edd_slg_social_type == 'vk' ) {
						$profile_pic_url = !empty($edd_slg_data['photo_big']) ? $edd_slg_data['photo_big'] : '';
					}
				}
			}
		}

		return $profile_pic_url;
	}

	/**
	 * Username Using Fname And Lname
	 * Handle to create username using api firstname and lastname
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.3.4
	 */
	public function edd_slg_username_by_fname_lname( $first_name = '', $last_name = '' ) {

		//Initilize username
		$username = '';
		$last_name_status = true;

		//If firstname is not empty
		if( !empty($first_name) ) {
			$username .= $first_name;
		}

		//If lastname is not empty
		if( !empty($first_name) && !empty($last_name) ) {
			$username .= '_' . $last_name;
			$last_name_status = false;
		}

		if( $last_name_status == true && !empty($last_name) ) {
			$username .= $last_name;
		}

		return apply_filters( 'edd_slg_username_by_fname_lname', $username, $first_name, $last_name );
	}

	/**
	 * Username Using Email
	 * Handle to create username using social email address
	 * 
	 * @package Easy Digital Downloads - Social Login
	 * @since 1.3.4
	 */
	public function edd_slg_username_by_email( $user_email = '' ) {

		//Initilize username
		$username = '';

		$username = str_replace( '@', '_', $user_email );
		$username = str_replace( '.', '_', $username );

		return $username;
	}

	/**
	 * Common Social Data Convertion
	 * 
	 * @package Easy Digital Download - Social Login
	 * @since 1.5.6
	 */
	public function edd_slg_get_user_common_social_data($social_data = array(), $social_type = '') {

		$common_social_data = array();

		// If social type is not empty
		if (!empty($social_type)) {
			switch ($social_type) {

				case 'facebook':
					$common_social_data['first_name'] = $social_data['first_name'];
					$common_social_data['last_name'] = $social_data['last_name'];
					$common_social_data['name'] = $social_data['name'];
					$common_social_data['email'] = $social_data['email'];
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = $social_data['link'];
					$common_social_data['id'] = $social_data['id'];
				break;

				case 'googleplus':

					if ( is_array( $social_data) ) {
						$firstname = isset($social_data["given_name"]) ? $social_data["given_name"] : '';
						$lastname = isset($social_data["family_name"]) ? $social_data["family_name"] : '';
						$name = isset($social_data["name"]) ? $social_data["name"] : '';
						$email = isset($social_data["email"]) ? $social_data["email"] : '';
						$id = isset($social_data["id"]) ? $social_data["id"] : '';
						$social_data['image']['url'] = isset($social_data["img_url"]) ? $social_data["img_url"] : '';
						$img_url = isset($social_data["img_url"]) ? $social_data["img_url"] : '';
					} else {

						$firstname = isset($social_data->given_name ) ? $social_data->given_name : '';
						$lastname = isset($social_data->family_name) ? $social_data->family_name : '';
						$name = isset($social_data->name) ? $social_data->name : '';
						$email = isset($social_data->email) ? $social_data->email : '';
						$id = isset($social_data->id) ? $social_data->id : '';
						$social_data['image']['url'] = isset($social_data->img_url) ? $social_data->img_url : '';
						$img_url = isset($social_data["img_url"]) ? $social_data["img_url"] : '';

					}

					$common_social_data['first_name'] = $firstname;
					$common_social_data['last_name'] = $lastname;
					$common_social_data['name'] = $name;
					$common_social_data['email'] = $email;
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = '';
					$common_social_data['id'] = $id;
					$common_social_data['image']['url'] = $img_url;
				break;

				case 'linkedin':
					$common_social_data['first_name'] = $social_data['firstName'];
					$common_social_data['last_name'] = $social_data['lastName'];
					$common_social_data['name'] = $social_data['firstName'] . ' ' . $social_data['lastName'];
					$common_social_data['email'] = $social_data['emailAddress'];
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = $social_data['publicProfileUrl'];
					$common_social_data['id'] = $social_data['id'];
				break;

				case 'yahoo':					
					$common_social_data['first_name'] = isset( $social_data['first_name'] ) ? $social_data['first_name'] : '';
					$common_social_data['last_name'] = isset( $social_data['last_name'] ) ? $social_data['last_name'] : '';
					$common_social_data['name'] = $common_social_data['first_name'].' '.$common_social_data['last_name'];
					$common_social_data['email'] = !empty( $social_data['email'] ) ? $social_data['email'] : '';
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = '';
					$common_social_data['id'] = isset( $social_data['identifier'] ) ? $social_data['identifier'] : '';
				break;

				case 'foursquare' :
					$common_social_data['first_name'] = $social_data->firstName;
					$common_social_data['last_name'] = $social_data->lastName;
					$common_social_data['name'] = $social_data->firstName . ' ' . $social_data->lastName;
					$common_social_data['email'] = $social_data->contact->email;
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = 'https://foursquare.com/user/' . $social_data->id;
					$common_social_data['id'] = $social_data->id;
				break;

				case 'windowslive' :
					$common_social_data['first_name'] = $social_data->first_name;
					$common_social_data['last_name'] = $social_data->last_name;
					$common_social_data['name'] = $social_data->name;
					$common_social_data['email'] = $social_data->wlemail;
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = $social_data->link;
					$common_social_data['id'] = $social_data->id;
				break;

				case 'vk' :
					$common_social_data['first_name'] = $social_data['first_name'];
					$common_social_data['last_name'] = $social_data['last_name'];
					$common_social_data['name'] = $social_data['first_name'] . ' ' . $social_data['last_name'];
					$common_social_data['email'] = isset($social_data['email']) ? $social_data['email'] : '';
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = EDD_SLG_VK_LINK . '/' . $social_data['screen_name'];
					$common_social_data['id'] = $social_data['uid'];
				break;

				case 'twitter' :
					$common_social_data['first_name'] = $social_data['name'];
					$common_social_data['last_name'] = '';
					$common_social_data['name'] = $social_data['screen_name']; //display name of user
					$common_social_data['type'] = 'twitter';
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = 'https://twitter.com/' . $social_data['screen_name'];
					$common_social_data['id'] = $social_data['id'];
					$common_social_data['email'] = (!empty($social_data['email']) ) ? $social_data['email'] : ''; // Twitter new option
				break;

				case 'amazon':
					$common_social_data['name'] = $social_data->name; //display name of user
					$common_social_data['id'] = $social_data->user_id;
					$common_social_data['email'] = $social_data->email;
					$common_social_data['all'] = $social_data;
					$common_social_data['type'] = $social_type;
				break;

				case 'paypal' :
					$common_social_data['first_name'] = isset($social_data->given_name) ? $social_data->given_name : '';
					$common_social_data['last_name'] = isset($social_data->family_name) ? $social_data->family_name : '';
					$common_social_data['email'] = $social_data->email;
					$common_social_data['name'] = isset($social_data->name) ? $social_data->name : ''; //display name of user
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['id'] = $social_data->user_id;
				break;

				case 'apple':
					$common_social_data['first_name'] = isset($social_data->name->firstName) ? $social_data->name->firstName : '';
					$common_social_data['last_name'] = isset($social_data->name->lastName
							) ? $social_data->name->lastName : '';
					$common_social_data['email'] = isset($social_data->email) ? $social_data->email : '';
					$common_social_data['name'] = isset($social_data->name->firstName) ? $social_data->name->firstName . ' ' . $social_data->name->lastName : '';
					$common_social_data['sub'] = isset($social_data->sub) ? $social_data->sub : ''; //display name of user
					$common_social_data['c_hash'] = isset($social_data->c_hash) ? $social_data->c_hash : '';
					$common_social_data['type'] = $social_type;
					$common_social_data['all'] = $social_data;
					$common_social_data['link'] = '';
					$common_social_data['sub'] = isset($social_data->sub) ? $social_data->sub : '';
				break;
			}
		}

		return apply_filters( 'edd_slg_get_user_common_social_data', $common_social_data, $social_type );
	}

}